<?php
/**
 * Plugin Name: Uronto Pay – WooCommerce Gateway
 * Plugin URI: https://pay.urontopay.com
 * Description: Pay via Uronto Pay (bKash, Nagad, Rocket etc.) using your Uronto Pay account.
 * Author: BlitheForge
 * Author URI: https://github.com/blitheforge
 * Version: 1.0.1
 * Requires at least: 5.2
 * Requires PHP: 7.2
 * Text Domain: urontopay
 */

if ( ! defined('ABSPATH') ) exit;

/**
 * Register after WooCommerce is loaded
 */
add_action('plugins_loaded', function () {
    if ( ! class_exists('WC_Payment_Gateway') ) return;

    class WC_UrontoPay_Gateway extends WC_Payment_Gateway {

        public function __construct() {
            $this->id                 = 'urontopay';
            $this->icon               = 'https://i.ibb.co/rRYWNb6m/20251006-171514.png';
            $this->has_fields         = false;
            $this->method_title       = __('Uronto Pay', 'urontopay');
            $this->method_description = __('Pay with Uronto Pay (bKash, Nagad, Rocket, etc.)', 'urontopay');

            $this->supports = array('products');

            $this->init_form_fields();
            $this->init_settings();

            // Settings
            $this->title         = $this->get_option('title');
            $this->description   = $this->get_option('description');
            $this->enabled       = $this->get_option('enabled');
            $this->api_key       = $this->get_option('api_key');
            $this->currency_rate = (float) $this->get_option('currency_rate');
            $this->is_digital    = $this->get_option('is_digital') === 'yes';
            $this->payment_site  = rtrim($this->get_option('payment_site'), '/'); // e.g. https://pay.urontopay.com

            // Save settings
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));

            // Webhook endpoint: ?wc-api=wc_urontopay_gateway
            add_action('woocommerce_api_' . strtolower(get_class($this)), array($this, 'handle_webhook'));
        }

        /**
         * Admin settings fields (mirrors Nagorikpay style)
         */
        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title'       => __('Enable/Disable', 'urontopay'),
                    'label'       => __('Enable Uronto Pay', 'urontopay'),
                    'type'        => 'checkbox',
                    'default'     => 'no',
                ),
                'title' => array(
                    'title'       => __('Title', 'urontopay'),
                    'type'        => 'text',
                    'description' => __('Title shown at checkout.', 'urontopay'),
                    'default'     => 'Uronto Pay Gateway',
                    'desc_tip'    => true,
                ),
                'description' => array(
                    'title'       => __('Description', 'urontopay'),
                    'type'        => 'textarea',
                    'default'     => __('Pay securely via Uronto Pay.', 'urontopay'),
                ),
                'api_key' => array(
                    'title'       => __('Enter API Key', 'urontopay'),
                    'type'        => 'text',
                    'description' => '',
                    'default'     => '',
                    'desc_tip'    => true,
                ),
                'currency_rate' => array(
                    'title'       => __('Enter USD Rate', 'urontopay'),
                    'type'        => 'number',
                    'description' => __('Used when store currency is USD.', 'urontopay'),
                    'default'     => '110',
                    'desc_tip'    => true,
                ),
                'is_digital' => array(
                    'title'       => __('Enable/Disable Digital product', 'urontopay'),
                    'label'       => __('Enable Digital product', 'urontopay'),
                    'type'        => 'checkbox',
                    'description' => __('If enabled, successful payment marks order Completed; otherwise Processing.', 'urontopay'),
                    'default'     => 'no',
                ),
                'payment_site' => array(
                    'title'             => __('Payment Site URL', 'urontopay'),
                    'type'              => 'text',
                    'description'       => '',
                    'default'           => 'https://pay.urontopay.com/',
                    'desc_tip'          => true,
                    'custom_attributes' => array('readonly' => 'readonly'),
                ),
            );
        }

        /**
         * Start a payment and redirect customer
         */
        public function process_payment( $order_id ) {
            $order = wc_get_order( $order_id );

            // Calculate amount (mirror logic but safer: use order total)
            $total    = (float) $order->get_total();
            $currency = $order->get_currency();

            if ( $currency === 'USD' && $this->currency_rate > 0 ) {
                $total = $total * $this->currency_rate;
            }

            // Build webhook URL (?wc-api=wc_urontopay_gateway&order_id=XYZ)
            $webhook_url = add_query_arg(
                array(
                    'wc-api'   => strtolower(get_class($this)),
                    'order_id' => $order->get_id(),
                ),
                home_url('/')
            );

            $payload = array(
                "cus_name"    => trim($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()),
                "cus_email"   => $order->get_billing_email(),
                "amount"      => round($total, 2),
                "webhook_url" => $webhook_url,
                "success_url" => $this->get_return_url($order),
                "cancel_url"  => wc_get_checkout_url(),
            );

            $endpoint = $this->payment_site . '/api/payment/create';
            $response = $this->remote_json_post( $endpoint, $payload );

            if ( is_wp_error($response) ) {
                wc_add_notice( __('Payment error: ', 'urontopay') . $response->get_error_message(), 'error' );
                return;
            }

            if ( isset($response['payment_url']) && ! empty($response['payment_url']) ) {
                if ( $order->get_status() !== 'completed' ) {
                    $order->update_status('pending', __('Customer is being redirected to Uronto Pay', 'urontopay'));
                }
                return array(
                    'result'   => 'success',
                    'redirect' => esc_url( $response['payment_url'] ),
                );
            }

            wc_add_notice( __('Unable to create payment. Please try again.', 'urontopay'), 'error' );
            return;
        }

        /**
         * Webhook handler (mirrors Nagorikpay flow: verify then set processing/completed)
         * Accepts JSON body or form POST. Needs order_id in query and transactionId/transaction_id in payload.
         */
        public function handle_webhook() {
            $order_id = isset($_GET['order_id']) ? absint($_GET['order_id']) : 0;
            $order    = $order_id ? wc_get_order($order_id) : false;

            // Accept both JSON and form-encoded
            $raw  = file_get_contents('php://input');
            $body = json_decode($raw, true);
            if ( ! is_array($body) ) $body = $_POST;

            $trx = '';
            if ( isset($body['transaction_id']) ) {
                $trx = sanitize_text_field($body['transaction_id']);
            } elseif ( isset($_REQUEST['transactionId']) ) { // compatibility
                $trx = sanitize_text_field($_REQUEST['transactionId']);
            }

            if ( ! $order || empty($trx) ) {
                status_header(400);
                wp_send_json(array('message' => 'Invalid data'));
            }

            // Verify on gateway
            $verify_endpoint = $this->payment_site . '/api/payment/verify';
            $verify = $this->remote_json_post( $verify_endpoint, array('transaction_id' => $trx) );

            if ( is_wp_error($verify) ) {
                status_header(502);
                wp_send_json(array('message' => $verify->get_error_message()));
            }

            // Expect something like: { status: "COMPLETED"/"PAID"/"PENDING", amount: ..., transaction_id: ... }
            $status = isset($verify['status']) ? strtoupper($verify['status']) : 'UNKNOWN';
            $amount = isset($verify['amount']) ? $verify['amount'] : '';
            $pm     = 'urontopay';

            if ( $order->get_status() !== 'completed' ) {
                if ( in_array($status, array('COMPLETED','PAID','SUCCESS')) ) {
                    $note = sprintf(
                        __('Uronto Pay payment was successfully %s. Payment Method: %s, Amount: %s, Transaction ID: %s, Email: %s', 'urontopay'),
                        strtolower($status), $pm, $amount, $trx, $order->get_billing_email()
                    );

                    if ( $this->is_digital ) {
                        // Completed for digital products
                        $order->update_status('completed', $note);
                        $order->reduce_order_stock();
                        $order->add_order_note( __('Payment completed via Uronto Pay URL checkout. trx id: ', 'urontopay') . $trx );
                        $order->payment_complete($trx);
                    } else {
                        // Processing for physical/mixed
                        $order->update_status('processing', $note);
                        $order->reduce_order_stock();
                        $order->payment_complete($trx);
                    }
                    status_header(200);
                    wp_send_json(array('ok' => 1, 'message' => 'Payment verified'));
                } else {
                    // On-hold if not found/ not completed
                    $order->update_status('on-hold', __('Uronto Pay payment on-hold. Transaction not completed yet. Please check manually.', 'urontopay'));
                    status_header(200);
                    wp_send_json(array('ok' => 0, 'message' => 'Not completed'));
                }
            }

            status_header(200);
            wp_send_json(array('ok' => 1, 'message' => 'Already completed'));
        }

        /**
         * Perform JSON POST with API key header
         */
        private function remote_json_post( $url, array $data ) {
            $resp = wp_remote_post( $url, array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'API-KEY'      => $this->api_key,
                ),
                'body'      => wp_json_encode($data),
                'timeout'   => 30,
                'sslverify' => true,
            ));

            if ( is_wp_error($resp) ) return $resp;

            $code = wp_remote_retrieve_response_code($resp);
            $body = wp_remote_retrieve_body($resp);
            $json = json_decode($body, true);

            if ( $code >= 200 && $code < 300 && is_array($json) ) {
                return $json;
            }
            return new WP_Error('urontopay_http', 'HTTP '.$code.' '.$body);
        }
    }

    // Register gateway with WooCommerce
    add_filter('woocommerce_payment_gateways', function($methods){
        $methods[] = 'WC_UrontoPay_Gateway';
        return $methods;
    });
});

/**
 * Optional REST endpoint (similar to Nagorikpay example)
 * POST /wp-json/urontopay/v1/webhook?order_id=123
 * Body: { "transaction_id": "..." }
 */
add_action('rest_api_init', function () {
    register_rest_route('urontopay/v1', '/webhook', array(
        'methods'  => 'POST',
        'callback' => function( WP_REST_Request $request ) {
            $order_id = absint( $request->get_param('order_id') );
            $trx      = sanitize_text_field( $request->get_param('transaction_id') ?: $request->get_param('transactionId') );

            if ( empty($order_id) || empty($trx) ) {
                return new WP_REST_Response( array('message' => 'Invalid data'), 400 );
            }

            $order = wc_get_order($order_id);
            if ( ! $order ) {
                return new WP_REST_Response( array('message' => 'Order not found'), 404 );
            }

            // Load gateway settings (so we can verify)
            $settings = get_option('woocommerce_urontopay_settings', array());
            $api_key  = isset($settings['api_key']) ? $settings['api_key'] : '';
            $site     = isset($settings['payment_site']) ? rtrim($settings['payment_site'], '/') : 'https://pay.urontopay.com';
            $is_digital = (isset($settings['is_digital']) && $settings['is_digital'] === 'yes');

            $resp = wp_remote_post( $site . '/api/payment/verify', array(
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'API-KEY'      => $api_key,
                ),
                'body'      => wp_json_encode(array('transaction_id' => $trx)),
                'timeout'   => 30,
                'sslverify' => true,
            ));

            if ( is_wp_error($resp) ) {
                return new WP_REST_Response( array('message' => $resp->get_error_message()), 502 );
            }

            $code = wp_remote_retrieve_response_code($resp);
            $body = json_decode( wp_remote_retrieve_body($resp), true );
            if ( $code < 200 || $code >= 300 || ! is_array($body) ) {
                return new WP_REST_Response( array('message' => 'Verify HTTP error'), 502 );
            }

            $status = isset($body['status']) ? strtoupper($body['status']) : 'UNKNOWN';
            $amount = isset($body['amount']) ? $body['amount'] : '';
            $pm     = 'urontopay';

            if ( $order->get_status() !== 'completed' && in_array($status, array('COMPLETED','PAID','SUCCESS')) ) {
                $note = sprintf(
                    __('Uronto Pay payment was successfully %s. Payment Method: %s, Amount: %s, Transaction ID: %s, Email: %s', 'urontopay'),
                    strtolower($status), $pm, $amount, $trx, $order->get_billing_email()
                );

                if ( $is_digital ) {
                    $order->update_status('completed', $note);
                    $order->reduce_order_stock();
                    $order->add_order_note( __('Payment completed via Uronto Pay URL checkout. trx id: ', 'urontopay') . $trx );
                    $order->payment_complete($trx);
                } else {
                    $order->update_status('processing', $note);
                    $order->reduce_order_stock();
                    $order->payment_complete($trx);
                }
                return new WP_REST_Response( array('ok' => 1), 200 );
            }

            if ( $order->get_status() === 'completed' ) {
                return new WP_REST_Response( array('ok' => 1, 'message' => 'Already completed'), 200 );
            }

            $order->update_status('on-hold', __('Uronto Pay payment on-hold. Transaction not completed yet. Please check manually.', 'urontopay'));
            return new WP_REST_Response( array('ok' => 0, 'message' => 'Not completed'), 200 );
        },
        'permission_callback' => '__return_true',
    ));
});